/* ====================================================================
 * Copyright (c) 1999-2017 The OpenSSL Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit. (http://www.OpenSSL.org/)"
 *
 * 4. The names "OpenSSL Toolkit" and "OpenSSL Project" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    openssl-core@OpenSSL.org.
 *
 * 5. Products derived from this software may not be called "OpenSSL"
 *    nor may "OpenSSL" appear in their names without prior written
 *    permission of the OpenSSL Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit (http://www.OpenSSL.org/)"
 *
 * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This product includes cryptographic software written by Eric Young
 * (eay@cryptsoft.com).  This product includes software written by Tim
 * Hudson (tjh@cryptsoft.com).
 *
 */

#include <stdint.h>
#include <stdlib.h>
#include <openssl/asn1.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/pkcs7.h>
#include <agwcl.h>

extern agwcl_error_t sign_with_device_key_pkcs1(unsigned char *data,
                         size_t data_len, unsigned char **signature,
                         size_t *signature_len, uint64_t flags);

extern uint64_t get_flags_encoding(uint64_t f);
extern uint64_t get_flags_key_alg(uint64_t f);
extern uint64_t get_flags_hash_alg(uint64_t f);

static int p7_type_is_other(PKCS7 *p7)
{
    int isOther = 1;

    int nid = OBJ_obj2nid(p7->type);

    switch (nid) {
    case NID_pkcs7_data:
    case NID_pkcs7_signed:
    case NID_pkcs7_enveloped:
    case NID_pkcs7_signedAndEnveloped:
    case NID_pkcs7_digest:
    case NID_pkcs7_encrypted:
        isOther = 0;
        break;
    default:
        isOther = 1;
    }

    return isOther;
}

static ASN1_OCTET_STRING *p7_get_octet_string(PKCS7 *p7)
{
    if (PKCS7_type_is_data(p7))
        return p7->d.data;
    if (p7_type_is_other(p7) && p7->d.other
        && (p7->d.other->type == V_ASN1_OCTET_STRING))
        return p7->d.other->value.octet_string;
    return NULL;
}

static BIO *p7_find_digest(EVP_MD_CTX **pmd, BIO *bio, int nid)
{
    for (;;) {
        bio = BIO_find_type(bio, BIO_TYPE_MD);
        if (bio == NULL) {
            PKCS7err(PKCS7_F_PKCS7_FIND_DIGEST,
                     PKCS7_R_UNABLE_TO_FIND_MESSAGE_DIGEST);
            return NULL;
        }
        BIO_get_md_ctx(bio, pmd);
        if (*pmd == NULL) {
            PKCS7err(PKCS7_F_PKCS7_FIND_DIGEST, ERR_R_INTERNAL_ERROR);
            return NULL;
        }
        if (EVP_MD_CTX_type(*pmd) == nid)
            return bio;
        bio = BIO_next(bio);
    }
    return NULL;
}

static agwcl_error_t p7_SIGNER_INFO_sign(PKCS7_SIGNER_INFO *si,
        uint64_t agwcl_flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    unsigned char *abuf = NULL;
    unsigned char *sigbuf = NULL;
    int alen;
    size_t siglen;
    uint64_t pkcs1_flags = get_flags_key_alg(agwcl_flags) |
                           get_flags_hash_alg(agwcl_flags) |
                           AGWCL_F_SIG_PKCS1 |
                           AGWCL_F_SIG_DER;

    alen = ASN1_item_i2d((ASN1_VALUE *)si->auth_attr, &abuf,
                         ASN1_ITEM_rptr(PKCS7_ATTR_SIGN));

    rc = sign_with_device_key_pkcs1(abuf, alen, &sigbuf, &siglen,
                pkcs1_flags);
    if (rc)
        goto err;

    ASN1_STRING_set0(si->enc_digest, sigbuf, siglen);

    rc = AGWCL_OK;;

err:
    return rc;
}

static agwcl_error_t p7_do_pkcs7_signed_attrib(PKCS7_SIGNER_INFO *si,
        EVP_MD_CTX *mctx, uint64_t agwcl_flags)
{
    agwcl_error_t rc = AGWCL_ERR_CREATE_PKCS7_ATTRIBUTE;
    unsigned char md_data[EVP_MAX_MD_SIZE];
    unsigned int md_len;

    /* Add signing time if not already present */
    if (!PKCS7_get_signed_attribute(si, NID_pkcs9_signingTime)) {
        if (!PKCS7_add0_attrib_signing_time(si, NULL))
            goto err;
    }

    /* Add digest */
    if (!EVP_DigestFinal_ex(mctx, md_data, &md_len)) {
        goto err;
    }
    if (!PKCS7_add1_attrib_digest(si, md_data, md_len)) {
        goto err;
    }

    /* Now sign the attributes */
    rc = p7_SIGNER_INFO_sign(si, agwcl_flags);

err:
    return rc;
}

static agwcl_error_t p7_data_final(PKCS7 *p7, BIO *bio, uint64_t agwcl_flags)
{
    agwcl_error_t rc = AGWCL_ERR_UNKNOWN;
    int i, j;
    BIO *btmp;
    PKCS7_SIGNER_INFO *si;
    EVP_MD_CTX *mdc, ctx_tmp;
    STACK_OF(X509_ATTRIBUTE) *sk;
    STACK_OF(PKCS7_SIGNER_INFO) *si_sk = NULL;
    ASN1_OCTET_STRING *os = NULL;

    if (p7 == NULL || p7->d.ptr == NULL)
        return AGWCL_INVALID_INPUT;

    EVP_MD_CTX_init(&ctx_tmp);
    i = OBJ_obj2nid(p7->type);
    p7->state = PKCS7_S_HEADER;

    switch (i) {
    case NID_pkcs7_data:
        os = p7->d.data;
        break;
    case NID_pkcs7_signedAndEnveloped:
        si_sk = p7->d.signed_and_enveloped->signer_info;
        os = p7->d.signed_and_enveloped->enc_data->enc_data;
        if (!os) {
            os = M_ASN1_OCTET_STRING_new();
            if (!os) {
                PKCS7err(PKCS7_F_PKCS7_DATAFINAL, ERR_R_MALLOC_FAILURE);
                goto err;
            }
            p7->d.signed_and_enveloped->enc_data->enc_data = os;
        }
        break;
    case NID_pkcs7_enveloped:
        os = p7->d.enveloped->enc_data->enc_data;
        if (!os) {
            os = M_ASN1_OCTET_STRING_new();
            if (!os) {
                PKCS7err(PKCS7_F_PKCS7_DATAFINAL, ERR_R_MALLOC_FAILURE);
                goto err;
            }
            p7->d.enveloped->enc_data->enc_data = os;
        }
        break;
    case NID_pkcs7_signed:
        si_sk = p7->d.sign->signer_info;
        os = p7_get_octet_string(p7->d.sign->contents);
        /* If detached data then the content is excluded */
        if (PKCS7_type_is_data(p7->d.sign->contents) && p7->detached) {
            M_ASN1_OCTET_STRING_free(os);
            os = NULL;
            p7->d.sign->contents->d.data = NULL;
        }
        break;

    case NID_pkcs7_digest:
        os = p7_get_octet_string(p7->d.digest->contents);
        /* If detached data then the content is excluded */
        if (PKCS7_type_is_data(p7->d.digest->contents) && p7->detached) {
            M_ASN1_OCTET_STRING_free(os);
            os = NULL;
            p7->d.digest->contents->d.data = NULL;
        }
        break;

    default:
        PKCS7err(PKCS7_F_PKCS7_DATAFINAL, PKCS7_R_UNSUPPORTED_CONTENT_TYPE);
        goto err;
    }

    if (si_sk != NULL) {
        for (i = 0; i < sk_PKCS7_SIGNER_INFO_num(si_sk); i++) {
            si = sk_PKCS7_SIGNER_INFO_value(si_sk, i);
            if (i > 0)
                continue;

            j = OBJ_obj2nid(si->digest_alg->algorithm);

            btmp = bio;

            btmp = p7_find_digest(&mdc, btmp, j);

            if (btmp == NULL)
                goto err;

            /*
             * We now have the EVP_MD_CTX, lets do the signing.
             */
            if (!EVP_MD_CTX_copy_ex(&ctx_tmp, mdc))
                goto err;

            sk = si->auth_attr;

            /*
             * If there are attributes, we add the digest attribute and only
             * sign the attributes
             */
            if (sk_X509_ATTRIBUTE_num(sk) > 0) {
                rc = p7_do_pkcs7_signed_attrib(si, &ctx_tmp,
                                               agwcl_flags);
                if (rc)
                    goto err;
            } else {
                goto err;
            }
        }
    } else if (i == NID_pkcs7_digest) {
        unsigned char md_data[EVP_MAX_MD_SIZE];
        unsigned int md_len;
        if (!p7_find_digest(&mdc, bio,
                               OBJ_obj2nid(p7->d.digest->md->algorithm)))
            goto err;
        if (!EVP_DigestFinal_ex(mdc, md_data, &md_len))
            goto err;
        M_ASN1_OCTET_STRING_set(p7->d.digest->digest, md_data, md_len);
    }

    if (!PKCS7_is_detached(p7)) {
        if (os == NULL)
            goto err;
        if (!(os->flags & ASN1_STRING_FLAG_NDEF)) {
            char *cont;
            long contlen;
            btmp = BIO_find_type(bio, BIO_TYPE_MEM);
            if (btmp == NULL) {
                PKCS7err(PKCS7_F_PKCS7_DATAFINAL,
                        PKCS7_R_UNABLE_TO_FIND_MEM_BIO);
                goto err;
            }
            contlen = BIO_get_mem_data(btmp, &cont);
            /*
             * Mark the BIO read only then we can use its copy of the data
             * instead of making an extra copy.
             */
            BIO_set_flags(btmp, BIO_FLAGS_MEM_RDONLY);
            BIO_set_mem_eof_return(btmp, 0);
            ASN1_STRING_set0(os, (unsigned char *)cont, contlen);
        }
    }

    rc = AGWCL_OK;

 err:
    EVP_MD_CTX_cleanup(&ctx_tmp);
    return rc;
}

static agwcl_error_t p7_final(PKCS7 *p7, BIO *data, int p7_flags,
        uint64_t agwcl_flags)
{
    agwcl_error_t rc = AGWCL_ERR_PKCS7_FORMAT;

    BIO *p7bio;
    p7bio = PKCS7_dataInit(p7, NULL);
    if (p7bio == NULL)
        goto err;

    SMIME_crlf_copy(data, p7bio, p7_flags);

    (void)BIO_flush(p7bio);

    rc = p7_data_final(p7, p7bio, agwcl_flags);
    if (rc)
        goto err;

    rc = AGWCL_OK;

 err:
    BIO_free_all(p7bio);

    return rc;
}

agwcl_error_t p7_sign_full(const EVP_MD *md, X509 *cert,
        unsigned char *data, size_t data_len,
        unsigned char **signature,  size_t *signature_len,
        uint64_t agwcl_flags)
{
    agwcl_error_t rc = AGWCL_NO_MEM;
    BIO *bin = NULL;
    BIO *bout = NULL;
    PKCS7 *p7 = NULL;
    PKCS7_SIGNER_INFO *si = NULL;
    X509_ALGOR *alg = NULL;
    const int out_buf_size = 10240;
    int p7_flags = PKCS7_PARTIAL | PKCS7_NOSMIMECAP | PKCS7_REUSE_DIGEST;

    *signature = realloc(*signature, out_buf_size);
    if (!*signature)
        goto err;

    bin = BIO_new_mem_buf(data, data_len);
    if (!bin)
        goto err;

    bout = BIO_new(BIO_s_mem());
    if (!bout)
        goto err;

    si = PKCS7_SIGNER_INFO_new();
    if (!si)
        goto err;

    rc = AGWCL_ERR_CREATE_PKCS7_ATTRIBUTE;

    p7 = PKCS7_sign(NULL, NULL, NULL, bin, p7_flags);
    if (!p7)
        goto err;

    /* We now need to add another PKCS7_SIGNER_INFO entry */
    if (!ASN1_INTEGER_set(si->version, 1))
        goto err;

    if (!X509_NAME_set(&si->issuer_and_serial->issuer,
                       X509_get_issuer_name(cert)))
        goto err;

    /*
     * because ASN1_INTEGER_set is used to set a 'long' we will do things the
     * ugly way.
     */
    M_ASN1_INTEGER_free(si->issuer_and_serial->serial);
    si->issuer_and_serial->serial =
              M_ASN1_INTEGER_dup(X509_get_serialNumber(cert));
    if (si->issuer_and_serial->serial == NULL)
        goto err;

    /* Set the algorithms */
    X509_ALGOR_set0(si->digest_alg, OBJ_nid2obj(EVP_MD_type(md)),
                    V_ASN1_NULL, NULL);

    PKCS7_SIGNER_INFO_get0_algs(si, NULL, NULL, &alg);
    if (alg)
        X509_ALGOR_set0(alg, OBJ_nid2obj(NID_rsaEncryption), V_ASN1_NULL, 0);

    if (!PKCS7_add_signer(p7, si))
        goto err;

    if (!PKCS7_add_certificate(p7, cert))
        goto err;

    if (!PKCS7_add_attrib_content_type(si, NULL))
        goto err;

    rc = p7_final(p7, bin, p7_flags, agwcl_flags);
    if (rc)
        goto err;

    switch (get_flags_encoding(agwcl_flags)) {
    case AGWCL_F_SIG_DER:
        if (!i2d_PKCS7_bio(bout, p7)) {
            rc = AGWCL_ERR_PKCS7_FORMAT;
            goto err;
        }
        break;
    case AGWCL_F_SIG_SMIME:
        if (!SMIME_write_PKCS7(bout, p7, bin, p7_flags)) {
            rc = AGWCL_ERR_PKCS7_FORMAT;
            goto err;
        }
        break;
    default:
        rc = AGWCL_ERR_FLAG_ENCODING;
    }

    *signature_len = BIO_read(bout, *signature, out_buf_size);

    rc = AGWCL_OK;

err:
    PKCS7_free(p7);
    BIO_free(bout);
    BIO_free(bin);

    return rc;
}
